/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	AboutPanel.cpp
	
	Contains:	Implementation of a last-minute about box.
	
	   Notes:	None.
	   
   ............................................................... */

#ifndef _ABOUT_PANEL_H
#include "AboutPanel.h"
#endif


/* ...............................................................
	String constants
	Strings for various components and subviews, which should
	probably be in resources.
   ............................................................... */

const uint8 NUM_ABOUT_TEXT_LINES = 12;
const char *ABOUT_PANEL_NAME = "AboutPanel";
const char *OK_BUTTON_TITLE = "OK";
const char *FEEDBACK_BUTTON_TITLE = "Feedback";
const char *EMAIL_MIME_TYPE = "text/x-email";
const char *FEEDBACK_ADDERSS = "mailto:moofie@pobox.com";
const char *HTML_MIME_TYPE = "text/html";
const char *HOME_PAGE_URL = "http://www.pobox.com/~moofie/beos/WindowColors";

const char *ABOUT_TEXT[NUM_ABOUT_TEXT_LINES] = 
		   {
				"WindowColors 1.0b1",
				"Copyright 1997-8 Steve Klingsporn & Marco Nelissen",
				HOME_PAGE_URL,
				"Shortcuts",
				"•  Use the Tab key to cycle through color labels.",
				"•  Use the arrow keys and numeric keypad to change " \
					"the selected color.",
				"•  Click on a model window or one of its labels to " \
					"activate it.",
				"•  Hold down the Option key to prevent changing "  \
					"the active model window.",
				"•  Click on a column heading to select the first " \
					"color label in that column.",
				"•  Right-drag from color labels and the color " \
					"picker to drag out a color.",
				"•  Color labels and the model window title tab " \
					"accept dropped colors.",
				"•  Use the Command key and the first letter of a " \
					"button to perform its action."
			};


/* ...............................................................
	AboutPanel constructor
	Stashes away our program icon as a bitmap.
   ............................................................... */

AboutPanel::AboutPanel(BRect frame)
		   :BView(frame, ABOUT_PANEL_NAME, 
		   				 B_FOLLOW_ALL_SIDES, 
		   				 B_WILL_DRAW),
		    _icon(NULL)
{
	/* Get our app icon, used in the about box */
	app_info appInfo;
	be_app->GetAppInfo(&appInfo);
	BFile file;
	file.SetTo(&appInfo.ref, B_READ_ONLY);
	BAppFileInfo appFileInfo;
	appFileInfo.SetTo(&file);
	_icon = new BBitmap(BRect(0, 0, 31, 31), B_COLOR_8_BIT);
	appFileInfo.GetIcon(_icon, B_LARGE_ICON);
}


/* ...............................................................
	AboutPanel destructor
	Deletes the program icon bitmap.
   ............................................................... */

AboutPanel::~AboutPanel()
{
	delete _icon;
}


/* ...............................................................
	AboutPanel::AttachedToWindow()
	Adds the subviews to this window and stuff.
   ............................................................... */

void AboutPanel::AttachedToWindow()
{
	SetViewColor(RGB_COLOR_216);
	
	/* Create the "OK" button */
	BButton *button =  new BButton(BRect(),
								   "",
								   OK_BUTTON_TITLE,
								   new BMessage(HIDE_ABOUT_PANEL));
	button->SetTarget(this);
	button->SetFlags(B_WILL_DRAW);
	if (be_plain_font->Size() > 14)
		button->SetFontSize(14);
	button->MakeDefault(true);
	button->ResizeToPreferred();
	BPoint origin = OK_BUTTON_RIGHT_EDGE;
	origin.x -= button->Frame().Width();
	button->MoveTo(origin);
	AddChild(button);
		
	/* Create the "Feedback" button */
	button = new BButton(BRect(),
						 "",
						 FEEDBACK_BUTTON_TITLE,
						 new BMessage(COMPOSE_FEEDBACK_MESSAGE));
	button->SetTarget(this);
	button->SetFlags(B_WILL_DRAW);
	if (be_plain_font->Size() > 14)
		button->SetFontSize(14);
	button->ResizeToPreferred();
	origin.x -= (10 + button->Frame().Width());
	origin.y += 3;
	button->MoveTo(origin);
	AddChild(button);
	
	/* Add some keyboard shortcuts */
	BMessage *message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", FEEDBACK_BUTTON_TITLE);
	Window()->AddShortcut('F', B_COMMAND_KEY,
						  message, this);
	
	message = new BMessage(INVOKE_BUTTON);
	message->AddString("which", OK_BUTTON_TITLE);
	Window()->AddShortcut('O', B_COMMAND_KEY,
						  message, this);
}


/* ...............................................................
	AboutPanel::Draw()
	Draws this view.  Not very well-optimized.
   ............................................................... */

void AboutPanel::Draw(BRect updateRect)
{
	BRect bounds = Bounds();
	BRect backgroundRect = bounds;
	backgroundRect.InsetBy(1, 1);
	
	/* Update the background */
	SetLowColor(RGB_COLOR_216);
	if (backgroundRect.Intersects(updateRect))
		FillRect((backgroundRect & updateRect), B_SOLID_LOW);
	
	/* Draw the bevel lines (no matter what) */
	BeginLineArray(4);
	AddLine(BPoint(bounds.left, bounds.top),
			BPoint(bounds.left, bounds.bottom),
			WHITE_COLOR);
	AddLine(BPoint(bounds.left + 1, bounds.top),
			BPoint(bounds.right, bounds.top),
			WHITE_COLOR);
	AddLine(BPoint(bounds.right, bounds.top + 1),
			BPoint(bounds.right, bounds.bottom),
			RGB_COLOR_192);
	AddLine(BPoint(bounds.right - 1, bounds.bottom),
			BPoint(bounds.left + 1, bounds.bottom),
			RGB_COLOR_192);
	EndLineArray();
	
	/* Draw the program icon */
	if (updateRect.Intersects(ICON_BOUNDS))
	{
		SetDrawingMode(B_OP_OVER);
		DrawBitmapAsync(_icon, _icon->Bounds(), ICON_BOUNDS);
		SetDrawingMode(B_OP_COPY);
	}
		
	/* Draw the program title */
	SetFont(be_bold_font);
	SetFontSize(14);
	BPoint origin = BPoint(82, 38);
	MovePenTo(origin);
	DrawString(ABOUT_TEXT[0]);
	
	/* Draw the copyright and contact info text.  We
	   should check each line against the updateRect,
	   and cache StringWidths to optimize, but I will
	   never ever ship this if I do everything right. */
	SetFont(be_plain_font);
	SetFontSize(10);

	for (uint8 i = 1; i < 3; i++)
	{
		origin.y += 14;
		MovePenTo(origin);
		if (i == 2) /* Why not blue? */
			SetHighColor(keyboard_navigation_color());
		DrawString(ABOUT_TEXT[i]);
	}
	
	/* Draw the "Shortcuts" line */
	origin.x = 30;
	origin.y += 45;
	SetFont(be_bold_font);
	SetFontSize(12);
	SetHighColor(BLACK_COLOR);
	MovePenTo(origin);
	DrawString(ABOUT_TEXT[3]);
	
	/* Draw the remainder of the about text lines */
	origin.y += 18;
	SetFont(be_plain_font);
	SetFontSize(10);
	
	for (uint8 i = 4; i < NUM_ABOUT_TEXT_LINES; i++)
	{
		MovePenTo(origin);
		DrawString(ABOUT_TEXT[i]);
		origin.y += 16;
	}
}


/* ...............................................................
	AboutPanel::MouseDown()
	Displays the WindowColors home page if the user hits in the
	supposed location of the link (generous).
   ............................................................... */

void AboutPanel::MouseDown(BPoint where)
{
	if (SUPPOSED_LINK_BOUNDS.Contains(where))
		VisitHomePage();
}


/* ...............................................................
	AboutPanel::MessageReceived()
	Handles messages for this view.
   ............................................................... */

void AboutPanel::MessageReceived(BMessage *message)
{
	switch (message->what)
	{
		/* Hide this panel (via "OK" button) */
		case HIDE_ABOUT_PANEL:
			((WindowColorsWindow *)Window())->
				ShowAboutPanel(false);
			break;
			
		/* Invoke a button (via keyboard shortcut) */
		case INVOKE_BUTTON:
		{
			const char *name;
			if (message->FindString("which", &name) == B_OK)
			{
				BButton *button = (BButton *)FindView(name);
				if (button)
				{
					button->MouseDown(BPoint(5, 5));
					button->Invoke();
				}
			}
		}
		break;
			
		/* Create an e-mail message (via "Feedback" button) */
		case COMPOSE_FEEDBACK_MESSAGE:
			ComposeFeedbackMessage();
			break;
			
		default:
			BView::MessageReceived(message);
			break;
	}
}


/* ...............................................................
	AboutPanel::Show()
	Makes sure our "OK" button is the default button.
   ............................................................... */

void AboutPanel::Show()
{
	((BButton *)ChildAt(0))->MakeDefault(true);
	BView::Show();
}


/* ...............................................................
	AboutPanel::ComposeFeedbackMessage()
	Launches the preferred application for the MIME type
	"text/x-email" with a B_ARGV_RECEIVED message that
	contains a URL specifying my e-mail address.
   ............................................................... */

void AboutPanel::ComposeFeedbackMessage()
{
	BMessage message = BMessage(B_ARGV_RECEIVED);
	message.AddInt32("argc", 2);
	message.AddString("argv", "");
	message.AddString("argv", FEEDBACK_ADDERSS);
	be_roster->Launch(EMAIL_MIME_TYPE, &message);
}


/* ...............................................................
	AboutPanel::VisitHomePage()
	Launches the preferred wubba * 3 application and passes it
	a B_ARGV_RECEIVED message with the URL of our home page.
   ............................................................... */
   
void AboutPanel::VisitHomePage()
{
	BMessage message = BMessage(B_ARGV_RECEIVED);
	message.AddInt32("argc", 2);
	message.AddString("argv", "");
	message.AddString("argv", HOME_PAGE_URL);
	be_roster->Launch(HTML_MIME_TYPE, &message);
}